<?php

namespace App\Controllers;

class MercadoLibreCron extends BaseController
{
    protected $mlModel;
    protected $configModel;

    public function __construct()
    {
        $this->mlModel = new \App\Models\MercadoLibreModel();
        $this->configModel = new \App\Models\ConfigModel();
        $this->mlController = new MercadoLibre();
    }

    private function refreshToken()
    {
        $clientId = $this->configModel->getConfig('ml_client_id')['valor'];
        $clientSecret = $this->configModel->getConfig('ml_client_secret')['valor'];
        $refreshToken = $this->configModel->getConfig('ml_refresh_token')['valor'];

        $ch = curl_init('https://api.mercadolibre.com/oauth/token');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
            'grant_type' => 'refresh_token',
            'client_id' => $clientId,
            'client_secret' => $clientSecret,
            'refresh_token' => $refreshToken
        ]));

        $response = curl_exec($ch);
        curl_close($ch);

        $data = json_decode($response, true);
        if (isset($data['access_token'])) {
            $this->configModel->setConfig('ml_access_token', $data['access_token']);
            $this->configModel->setConfig('ml_refresh_token', $data['refresh_token']);
            return $data['access_token'];
        }

        throw new \Exception('Error al refrescar el token');
    }

    private function actualizarStock($producto, $accessToken)
    {
        try {
            // Primero obtenemos el user ID del vendedor
            $urlUser = "https://api.mercadolibre.com/users/me";
            $ch = curl_init($urlUser);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken
            ]);

            $responseUser = curl_exec($ch);
            $userData = json_decode($responseUser, true);

            if (!isset($userData['id'])) {
                throw new \Exception("No se pudo obtener la información del usuario");
            }

            $userId = $userData['id'];
            // Primero intentamos buscar por seller_custom_field (SKU)
            $urlSearchCustom = "https://api.mercadolibre.com/users/{$userId}/items/search?sku=" . urlencode($producto['CodProd']);

            $ch = curl_init($urlSearchCustom);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken
            ]);

            $responseSearch = curl_exec($ch);
            $searchData = json_decode($responseSearch, true);

            // // Si no encontramos por seller_custom_field, intentamos con seller_sku
            if (empty($searchData['results'])) {
                $urlSearchSku = "https://api.mercadolibre.com/users/{$userId}/items/search?seller_sku=" . urlencode($producto['CodProd']);

                curl_setopt($ch, CURLOPT_URL, $urlSearchSku);
                $responseSearch = curl_exec($ch);
                $searchData = json_decode($responseSearch, true);
            }

            // // Si no encontramos el producto, registramos el error
            if (empty($searchData['results'])) {
                throw new \Exception("No se encontró el producto con SKU: " . $producto['CodProd']);
            }

            // // Obtenemos el primer resultado (debería ser único por SKU)
            $itemId = $searchData['results'][0];

            // // Obtenemos los detalles del item
            $urlItem = "https://api.mercadolibre.com/items/" . $itemId;
            curl_setopt($ch, CURLOPT_URL, $urlItem);
            $responseItem = curl_exec($ch);
            $itemData = json_decode($responseItem, true);

            if (!isset($itemData['id'])) {
                 throw new \Exception("Error al obtener detalles del producto de ML");
            }

            // // Actualizamos el stock
            $statusPub = $itemData['status'];

            if(intval($producto['Existen']) == 0) {
                $statusPub = 'paused';
            }

            if(intval($producto['Existen']) > 0 && $itemData['status'] == 'paused') {
                $statusPub = 'active';
            }
            $updateData = [
                'available_quantity' => (int) $producto['Existen'],
                'status' => (string) $statusPub
            ];

            curl_setopt($ch, CURLOPT_URL, $urlItem);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($updateData));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken,
                'Content-Type: application/json'
            ]);

            $responseUpdate = curl_exec($ch);
            $updateData = json_decode($responseUpdate, true);
            curl_close($ch);

            if (isset($updateData['error'])) {
                throw new \Exception("Error al actualizar stock: " . ($updateData['message'] ?? 'Error desconocido'));
            }

            // // Registrar sincronización exitosa
            $mlModel = new \App\Models\MercadoLibreModel();
            $mlModel->registrarSincronizacion(
                $producto['CodProd'],
                $itemData['id'],
                1
            );

            return [
                'success' => true,
                'response' => $updateData,
                'producto' => $producto,
                'message' => 'Stock actualizado correctamente',
                'data' => [
                    'ml_id' => $itemData['id'],
                    'sku' => $producto['CodProd'],
                    'title' => $itemData['title'],
                    'status' => $statusPub,
                    'new_stock' => (int) $producto['Existen']
                ]
            ];

            //return $searchData['results'];

        } catch (\Exception $e) {
            // Registrar error en sincronización
            // if (isset($mlModel)) {
            //     $mlModel->registrarSincronizacion(
            //         $producto['CodProd'],
            //         null,
            //         0
            //     );
            // }

            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function syncMercadoLibre()
    {
        try {
            $mlModel = new \App\Models\MercadoLibreModel();
            $data = $mlModel->syncMercadoLibreData();
            log_message('info', 'Sincronización con MercadoLibre completada exitosamente');
            return $this->response->setJSON(['success' => true, 'data' => $data]);
        } catch (\Exception $e) {
            log_message('error', 'Error en sincronización con MercadoLibre: ' . $e->getMessage());
            return $this->response->setJSON(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    public function syncProducts()
    {
        try {
            $db = \Config\Database::connect();
            
            // Obtener productos con SINCRONIZACION activa
            $query = $db->table('SAPROD_02 p2')
                       ->select('p2.CodProd, p2.SINCRONIZACION, ml.ml_id, ml.selected')
                       ->join('ml_productos_sync ml', 'p2.CodProd = ml.cod_prod', 'left')
                       ->where('p2.SINCRONIZACION', 1);
            
            $products = $query->get()->getResultArray();

            foreach ($products as $product) {
                // Actualizar selected en ml_productos_sync
                if ($product['ml_id']) {
                    //$this->mlModel->update(
                    //    ['ml_id' => $product['ml_id']],
                    //    ['selected' => $product['SINCRONIZACION']]
                    //);
                    $this->mlModel->where('ml_id', $product['ml_id'])
                     ->set(['selected' => $product['SINCRONIZACION']])
                     ->update();
                }
            }

            // Sincronizar productos seleccionados con MercadoLibre
            if ($this->configModel->getConfig('ml_sync_active')['valor']) {
                $selectedProducts = $this->mlModel->where('selected', 1)->findAll();
                $accessToken = $this->refreshToken();

                foreach ($selectedProducts as $product) {
                    $this->actualizarStock($product, $accessToken);
                }
            }

            log_message('info', 'Cron job de sincronización completado exitosamente');
            return $this->response->setJSON(['success' => true, 'Productos actualizados' => $selectedProducts]);

        } catch (\Exception $e) {
            log_message('error', 'Error en cron job de sincronización: ' . $e->getMessage());
            return $this->response->setJSON(['success' => false, 'error' => $e->getMessage()]);
        }
    }
}