<?php

namespace App\Models;

use CodeIgniter\Model;

class MercadoLibreOrdersModel extends Model
{
    protected $table = 'ml_orders';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $allowedFields = [
        'order_id', 'date_created', 'date_closed', 'status',
        'total_amount', 'currency_id', 'buyer_id', 'buyer_nickname',
        'shipping_id', 'shipping_status', 'items_count', 'payment_method',
        'order_json', 'processed', 'invoice_id', 'fecha_registro'
    ];

    protected $useTimestamps = false;
    
    public function __construct()
    {
        parent::__construct();
        $this->createTableIfNotExists();
    }
    
    private function createTableIfNotExists()
    {
        $forge = \Config\Database::forge();
        
        if (!$this->db->tableExists($this->table)) {
            $forge->addField([
                'id' => [
                    'type' => 'INT',
                    'constraint' => 11,
                    'unsigned' => true,
                    'auto_increment' => true
                ],
                'order_id' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                ],
                'date_created' => [
                    'type' => 'DATETIME',
                ],
                'date_closed' => [
                    'type' => 'DATETIME',
                    'null' => true,
                ],
                'status' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                ],
                'total_amount' => [
                    'type' => 'DECIMAL',
                    'constraint' => '10,2',
                ],
                'currency_id' => [
                    'type' => 'VARCHAR',
                    'constraint' => 10,
                ],
                'buyer_id' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                ],
                'buyer_nickname' => [
                    'type' => 'VARCHAR',
                    'constraint' => 100,
                ],
                'shipping_id' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true,
                ],
                'shipping_status' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true,
                ],
                'items_count' => [
                    'type' => 'INT',
                    'constraint' => 5,
                ],
                'payment_method' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true,
                ],
                'order_json' => [
                    'type' => 'TEXT',
                ],
                'processed' => [
                    'type' => 'TINYINT',
                    'constraint' => 1,
                    'default' => 0,
                ],
                'invoice_id' => [
                    'type' => 'VARCHAR',
                    'constraint' => 50,
                    'null' => true,
                ],
                'fecha_registro' => [
                    'type' => 'DATETIME',
                    'default' => null,
                ],
            ]);
            
            $forge->addKey('id', true);
            $forge->addKey('order_id');
            $forge->createTable($this->table);
        }
    }
    
    /**
     * Obtiene las órdenes paginadas para DataTables
     */
    public function getOrdersPaginated($start, $length, $search, $orderColumn, $orderDir)
    {
        $builder = $this->db->table($this->table);
        
        // Total de registros sin filtro
        $totalRecords = $builder->countAllResults(false);
        
        // Aplicar búsqueda si existe
        if (!empty($search)) {
            $builder->groupStart()
                ->like('order_id', $search)
                ->orLike('buyer_nickname', $search)
                ->orLike('status', $search)
            ->groupEnd();
        }
        
        // Total de registros con filtro
        $recordsFiltered = $builder->countAllResults(false);
        
        // Ordenar y paginar
        $builder->orderBy($orderColumn, $orderDir)
                ->limit($length, $start);
                
        $results = $builder->get()->getResultArray();
        
        return [
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $recordsFiltered,
            'data' => $results
        ];
    }
    
    /**
     * Guarda una orden de MercadoLibre en la base de datos
     */
    public function saveOrder($orderData)
    {
        // Verificar si la orden ya existe
        $existingOrder = $this->where('order_id', $orderData['order_id'])->first();
        
        if ($existingOrder) {
            // Actualizar orden existente
            $this->update($existingOrder['id'], $orderData);
            return $existingOrder['id'];
        } else {
            // Crear nueva orden
            return $this->insert($orderData);
        }
    }
    
    /**
     * Obtiene los detalles de una orden específica
     */
    public function getOrderDetails($orderId)
    {
        return $this->where('order_id', $orderId)->first();
    }
    
    /**
     * Marcar una orden como procesada con su factura
     */
    public function markAsProcessed($orderId, $invoiceId)
    {
        return $this->where('order_id', $orderId)
                    ->set(['processed' => 1, 'invoice_id' => $invoiceId])
                    ->update();
    }
}