<?php

namespace App\Models;

use CodeIgniter\Model;

class ProveedorModel extends Model
{
    protected $table = 'SAPROV';
    protected $primaryKey = 'CodProv';
    protected $useAutoIncrement = false;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'CodProv',
        'Descrip',
        'Direc1',
        'Direc2',
        'Ciudad',
        'Estado',
        'Pais',
        'ZipCode',
        'Telef',
        'Fax',
        'Email',
        'ContactP',
        'ID3',
        'Activo',
        'EsProvee',
        'DiasC',
        'LimitC',
        'SaldoAct',
        'MontoVtas',
        'FechaUV',
        'FechaUP',
        'CodVend',
        'DesctoP',
        'FechaI',
        'TipoID',
        'Notas'
    ];

    protected $useTimestamps = false;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    protected $validationRules = [
        'CodProv' => 'required|max_length[15]',
        'Descrip' => 'required|max_length[60]'
    ];

    protected $validationMessages = [
        'CodProv' => [
            'required' => 'El código del proveedor es requerido',
            'max_length' => 'El código del proveedor no puede exceder 15 caracteres'
        ],
        'Descrip' => [
            'required' => 'La descripción del proveedor es requerida',
            'max_length' => 'La descripción no puede exceder 60 caracteres'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    /**
     * Obtener todos los proveedores activos
     */
    public function getProveedores()
    {
        return $this->where('Activo', 1)
                   ->where('EsProvee', 1)
                   ->orderBy('Descrip', 'ASC')
                   ->findAll();
    }

    /**
     * Obtener un proveedor por código
     */
    public function getProveedor($codProv)
    {
        return $this->where('CodProv', $codProv)->first();
    }

    /**
     * Buscar proveedores por término
     */
    public function buscarProveedores($termino)
    {
        return $this->groupStart()
                   ->like('CodProv', $termino)
                   ->orLike('Descrip', $termino)
                   ->orLike('ID3', $termino)
                   ->groupEnd()
                   ->where('Activo', 1)
                   ->where('EsProvee', 1)
                   ->orderBy('Descrip', 'ASC')
                   ->findAll();
    }

    /**
     * Obtener proveedor por RUT
     */
    public function getProveedorPorRut($rut)
    {
        return $this->where('ID3', $rut)
                   ->where('Activo', 1)
                   ->first();
    }

    /**
     * Crear nuevo código de proveedor
     */
    public function generarCodigoProveedor()
    {
        $db = \Config\Database::connect();
        
        // Obtener el siguiente correlativo
        $correlativo = $db->table('SACORRELSIS')
            ->where('FieldName', 'PrxProv')
            ->get()
            ->getRowArray();
        
        if ($correlativo) {
            $numero = intval($correlativo['ValueInt']) + 1;
            
            // Actualizar el correlativo
            $db->table('SACORRELSIS')
                ->where('FieldName', 'PrxProv')
                ->update(['ValueInt' => $numero]);
            
            return 'P' . str_pad($numero, 6, '0', STR_PAD_LEFT);
        }
        
        return null;
    }

    /**
     * Validar si existe un proveedor con el mismo RUT
     */
    public function existeRut($rut, $excluirCodigo = null)
    {
        $query = $this->where('ID3', $rut);
        
        if ($excluirCodigo) {
            $query->where('CodProv !=', $excluirCodigo);
        }
        
        return $query->countAllResults() > 0;
    }

    /**
     * Obtener estadísticas de compras por proveedor
     */
    public function getEstadisticasCompras($codProv, $fechaDesde = null, $fechaHasta = null)
    {
        $db = \Config\Database::connect();
        
        $query = $db->table('SACOMP')
            ->select('
                COUNT(*) as total_compras,
                SUM(MtoTotal) as monto_total,
                AVG(MtoTotal) as promedio_compra,
                MAX(FechaE) as ultima_compra
            ')
            ->where('CodProv', $codProv)
            ->where('TipoCom', 'A')
            ->where('Status', 'A');
        
        if ($fechaDesde) {
            $query->where('FechaE >=', date('m-d-Y', strtotime($fechaDesde)));
        }
        
        if ($fechaHasta) {
            $query->where('FechaE <=', date('m-d-Y', strtotime($fechaHasta)));
        }
        
        return $query->get()->getRowArray();
    }
}