<?php

namespace App\Controllers;

class Archivos extends BaseController
{
    protected $clientesModel;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
        $this->clientesModel = new \App\Models\ClienteModel();
    }

    /**
     * Dashboard principal de archivos
     */
    public function index()
    {
        // Obtener estadísticas para el dashboard
        $data = [
            'total_clientes' => $this->clientesModel->countAll(),
            'total_convenios' => 0, // Implementar cuando tengas el modelo
            'total_servidores' => 0, // Implementar cuando tengas el modelo
            'total_retenciones' => 1, // Valor fijo como en la imagen
            'total_finalizar' => 0, // Implementar según tu lógica
        ];

        return view('archivos/dashboard', $data);
    }

    /**
     * Mostrar listado de clientes en la sección archivos
     */
    public function clientes()
    {
        $clientes = $this->clientesModel->getClientes(50);
        
        // Calcular estadísticas
        $total_clientes = count($clientes);
        $clientes_activos = count(array_filter($clientes, function($c) { 
            return $c['Activo'] == 1; 
        }));
        $clientes_inactivos = $total_clientes - $clientes_activos;
        
        // Clientes nuevos este mes (necesitarás ajustar según tu campo de fecha)
        $clientes_mes = 0; // Implementar lógica según tengas campo created_at

        $data = [
            'clientes' => $clientes,
            'total_clientes' => $total_clientes,
            'clientes_activos' => $clientes_activos,
            'clientes_inactivos' => $clientes_inactivos,
            'clientes_mes' => $clientes_mes
        ];

        return view('archivos/clientes', $data);
    }

    private function getPaises()
    {
        try {
            $query = $this->db->query("SELECT TOP (1000) [Descrip], [Pais] FROM [Ftodo].[dbo].[SAPAIS] ORDER BY [Descrip]");
            return $query->getResultArray();
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener países: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Obtener estados por país via AJAX
     */
    public function getEstados()
    {
        $paisId = $this->request->getVar('pais_id');
        
        if (!$paisId) {
            return $this->response->setJSON(['error' => 'País no especificado']);
        }
        
        try {
            $query = $this->db->query(
                "SELECT TOP (1000) [Descrip], [Estado], [Pais] FROM [Ftodo].[dbo].[SAESTADO] WHERE [Pais] = ? ORDER BY [Descrip]", 
                [$paisId]
            );
            $estados = $query->getResultArray();
            
            return $this->response->setJSON($estados);
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener estados: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Error al cargar estados']);
        }
    }
    
    /**
     * Obtener ciudades por estado via AJAX
     */
    public function getCiudades()
    {
        $estadoId = $this->request->getVar('estado_id');
        
        if (!$estadoId) {
            return $this->response->setJSON(['error' => 'Estado no especificado']);
        }
        
        try {
            $query = $this->db->query(
                "SELECT TOP (1000) [Descrip], [Ciudad], [Estado] FROM [Ftodo].[dbo].[SACIUDAD] WHERE [Estado] = ? ORDER BY [Descrip]", 
                [$estadoId]
            );
            $ciudades = $query->getResultArray();
            
            return $this->response->setJSON($ciudades);
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener ciudades: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Error al cargar ciudades']);
        }
    }
    
    /**
     * Obtener municipios por ciudad via AJAX (opcional)
     */
    public function getMunicipios()
    {
        $ciudadId = $this->request->getVar('ciudad_id');
        
        if (!$ciudadId) {
            return $this->response->setJSON(['error' => 'Ciudad no especificada']);
        }
        
        try {
            // Si tienes una tabla de municipios, usa esta consulta
            // Sino, puedes retornar la misma ciudad como municipio
            $municipios = [
                ['Descrip' => 'Centro', 'Municipio' => $ciudadId . '_CENTRO'],
                ['Descrip' => 'Norte', 'Municipio' => $ciudadId . '_NORTE'],
                ['Descrip' => 'Sur', 'Municipio' => $ciudadId . '_SUR'],
                ['Descrip' => 'Oriente', 'Municipio' => $ciudadId . '_ORIENTE'],
                ['Descrip' => 'Occidente', 'Municipio' => $ciudadId . '_OCCIDENTE']
            ];
            
            return $this->response->setJSON($municipios);
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener municipios: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Error al cargar municipios']);
        }
    }
    
    /**
     * Obtener estados por defecto (Colombia)
     */
    private function getEstadosPorDefecto($paisId = 2) // Asumiendo que Colombia tiene ID 2
    {
        try {
            $query = $this->db->query(
                "SELECT TOP (1000) [Descrip], [Estado], [Pais] FROM [Ftodo].[dbo].[SAESTADO] WHERE [Pais] = ? ORDER BY [Descrip]", 
                [$paisId]
            );
            return $query->getResultArray();
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener estados por defecto: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Crear nuevo cliente desde archivos
     */
    public function crear_cliente()
    {
        $ultimoCodigo = $this->db->table('SACLIE')
            ->selectMax('CodClie')
            ->where('CodClie LIKE', 'C%')
            ->get()
            ->getRowArray();
            
        $ultimoNumero = 0;
        if ($ultimoCodigo && isset($ultimoCodigo['CodClie'])) {
            $ultimoNumero = intval(substr($ultimoCodigo['CodClie'], 1));
        }
        $proximoCodigo = 'C' . str_pad(($ultimoNumero + 1), 5, '0', STR_PAD_LEFT);

        $data = [
            'ultimo_codigo' => $proximoCodigo,
            'paises' => $this->getPaises(),
            'estados' => $this->getEstadosPorDefecto(), // Cargar estados por defecto (ej: Colombia)
            'ciudades' => [],
            'municipios' => [],
        ];

        return view('archivos/clientes_crear', $data);
    }

    /**
     * Guardar nuevo cliente
     */
    public function guardar_cliente()
    {
        $validation = \Config\Services::validation();
        
        $validation->setRules([
            'CodClie' => 'required|max_length[20]|is_unique[clientes.CodClie]',
            'Descrip' => 'required|max_length[100]',
            'ID3' => 'required|max_length[20]',
            'Telef' => 'permit_empty|max_length[20]',
            'Email' => 'permit_empty|valid_email|max_length[100]',
            'Activo' => 'required|in_list[0,1]'
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        $data = [
            'CodClie' => $this->request->getPost('CodClie'),
            'Descrip' => $this->request->getPost('Descrip'),
            'ID3' => $this->request->getPost('ID3'),
            'Telef' => $this->request->getPost('Telef'),
            'Email' => $this->request->getPost('Email'),
            'Activo' => $this->request->getPost('Activo')
        ];

        if ($this->clientesModel->insert($data)) {
            return redirect()->to('/archivos/clientes')->with('message', 'Cliente creado exitosamente');
        } else {
            return redirect()->back()->withInput()->with('error', 'Error al crear el cliente');
        }
    }

    /**
     * Ver detalles de un cliente
     */
    public function ver_cliente($codigo)
    {
        $cliente = $this->clientesModel->where('CodClie', $codigo)->first();
        
        if (!$cliente) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Cliente no encontrado');
        }

        $data = ['cliente' => $cliente];
        return view('archivos/clientes_ver', $data);
    }

    /**
     * Editar cliente
     */
    public function editar_cliente($codigo)
    {
        $cliente = $this->clientesModel->where('CodClie', $codigo)->first();
        
        if (!$cliente) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Cliente no encontrado');
        }

        $data = ['cliente' => $cliente];
        return view('archivos/clientes_editar', $data);
    }

    /**
     * Actualizar cliente
     */
    public function actualizar_cliente($codigo)
    {
        $cliente = $this->clientesModel->where('CodClie', $codigo)->first();
        
        if (!$cliente) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Cliente no encontrado');
        }

        $validation = \Config\Services::validation();
        
        $validation->setRules([
            'Descrip' => 'required|max_length[100]',
            'ID3' => 'required|max_length[20]',
            'Telef' => 'permit_empty|max_length[20]',
            'Email' => 'permit_empty|valid_email|max_length[100]',
            'Activo' => 'required|in_list[0,1]'
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        $data = [
            'Descrip' => $this->request->getPost('Descrip'),
            'ID3' => $this->request->getPost('ID3'),
            'Telef' => $this->request->getPost('Telef'),
            'Email' => $this->request->getPost('Email'),
            'Activo' => $this->request->getPost('Activo')
        ];

        if ($this->clientesModel->where('CodClie', $codigo)->set($data)->update()) {
            return redirect()->to('/archivos/clientes')->with('message', 'Cliente actualizado exitosamente');
        } else {
            return redirect()->back()->withInput()->with('error', 'Error al actualizar el cliente');
        }
    }

    /**
     * Eliminar cliente
     */
    public function eliminar_cliente($codigo)
    {
        $cliente = $this->clientesModel->where('CodClie', $codigo)->first();
        
        if (!$cliente) {
            return redirect()->to('/archivos/clientes')->with('error', 'Cliente no encontrado');
        }

        if ($this->clientesModel->where('CodClie', $codigo)->delete()) {
            return redirect()->to('/archivos/clientes')->with('message', 'Cliente eliminado exitosamente');
        } else {
            return redirect()->to('/archivos/clientes')->with('error', 'Error al eliminar el cliente');
        }
    }

    /**
     * Secciones adicionales del dashboard de archivos
     */
    public function convenios()
    {
        // Implementar cuando tengas el modelo de convenios
        $data = ['convenios' => []];
        return view('archivos/convenios', $data);
    }

    public function servidores()
    {
        // Implementar cuando tengas el modelo de servidores
        $data = ['servidores' => []];
        return view('archivos/servidores', $data);
    }

    public function retenciones()
    {
        // Implementar cuando tengas el modelo de retenciones
        $data = ['retenciones' => []];
        return view('archivos/retenciones', $data);
    }

    public function finalizar()
    {
        // Implementar según tu lógica de negocio
        $data = ['items_finalizar' => []];
        return view('archivos/finalizar', $data);
    }
}