<?php

namespace App\Controllers;

class MercadoLibre extends BaseController
{
    protected $configModel;
    protected $mlModel;

    protected $productosModel;

    public function __construct()
    {
        $this->configModel = new \App\Models\ConfigModel();
        $this->mlModel = new \App\Models\MercadoLibreModel();
        $this->productosModel = new \App\Models\ProductosModel();
    }

    public function index()
    {
        $data['ml_config'] = [
            'client_id' => $this->configModel->getConfig('ml_client_id'),
            'client_secret' => $this->configModel->getConfig('ml_client_secret'),
            'sync_active' => $this->configModel->getConfig('ml_sync_active')
        ];

        return view('home_mercadolibre', $data);
    }

    public function toggleSync()
    {
        if ($this->request->isAJAX()) {
            $estado = $this->request->getPost('estado');
            $this->configModel->setConfig('ml_sync_active', $estado);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Estado de sincronización actualizado'
            ]);
        }
    }
    
    public function listarProductosML()
    {
        try {
            $accessToken = $this->refreshToken();
            // Obtener el user ID del vendedor
            $urlUser = "https://api.mercadolibre.com/users/me";
            $ch = curl_init($urlUser);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken
            ]);
            $responseUser = curl_exec($ch);
            $userData = json_decode($responseUser, true);
            
            if (!isset($userData['id'])) {
                throw new \Exception("No se pudo obtener la información del usuario");
            }
            
            $userId = $userData['id'];
            $allItems = [];
            $allScrollID = [];
            $lastpaging = 0;
            
            // Primer intento usando scroll para cuentas con más de 1000 productos
            $urlItems = "https://api.mercadolibre.com/users/{$userId}/items/search?" .
                "search_type=scan&" .
                "limit=100&" .
                "status=active,paused,closed,inactive";
                
            curl_setopt($ch, CURLOPT_URL, $urlItems);
            $responseItems = curl_exec($ch);
            $itemsData = json_decode($responseItems, true);

            log_message('info', 'Resultado Compuesto: ' . json_encode($itemsData));
            
            // Si obtenemos scroll_id, significa que hay más de 1000 productos
            if (isset($itemsData['scroll_id'])) {
                $scrollId = $itemsData['scroll_id'];
                $totalProducts = $itemsData['paging']['total'];
                $processedCount = 0;
                $iterationCount = 0; // Contador de iteraciones
                $maxIterations = 30; // Límite de iteraciones para testing
                
                do {
                    // Procesar los resultados actuales
                                       
                    
                    // Preparar la siguiente consulta con scroll_id
                    $urlScroll = "https://api.mercadolibre.com/users/{$userId}/items/search?" .
                        "search_type=scan&limit=100&" .
                        "scroll_id=" . $scrollId;
                        
                    curl_setopt($ch, CURLOPT_URL, $urlScroll);
                    $responseItems2 = curl_exec($ch);
                    $itemsData2 = json_decode($responseItems2, true);

                    $allScrollID = array_merge($allScrollID, $itemsData2['results']);

                    $scrollId = $itemsData2['scroll_id'];

                    $lastpaging = $itemsData2['paging']['limit'];

                    if (isset($itemsData2['results'])) {
                        $allItems = array_merge($allItems, $this->processItems($ch, $itemsData2['results']));
                    }

                    $iterationCount++;
                    $processedCount += count($itemsData2['results']);
                    

                    if ($processedCount >= $totalProducts) {
                        break;
                    }

                    usleep(100000);
                    
                } while (true);
             
            } else {
            //     // Método original para menos de 1000 productos
                $offset = 0;
                $limit = 50;
                
                do {
                    $urlItems = "https://api.mercadolibre.com/users/{$userId}/items/search?" .
                        "offset={$offset}&" .
                        "limit={$limit}&" .
                        "status=active,paused,closed,inactive";
                        
                    curl_setopt($ch, CURLOPT_URL, $urlItems);
                    $responseItems = curl_exec($ch);
                    $itemsData = json_decode($responseItems, true);
                    
                    if (!isset($itemsData['results'])) {
                        break;
                    }
                    
                    $allItems = array_merge($allItems, $this->processItems($ch, $itemsData['results']));
                    $offset += $limit;
                    
                } while (count($itemsData['results']) == $limit);
            }
            
            curl_close($ch);
            
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Productos obtenidos correctamente',
                'total' => count($allItems),
                'scroll_init' => $scrollId,
                'scroll_id' => $allScrollID,
                'lastpaging' => $lastpaging,
                'items' => $allItems
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error al obtener productos: ' . $e->getMessage()
            ]);
        }
    }

    // Método auxiliar para procesar los items
    private function processItems($ch, $itemIds)
    {
        $processedItems = [];
        
        foreach ($itemIds as $itemId) {
            $urlItemDetail = "https://api.mercadolibre.com/items/{$itemId}";
            curl_setopt($ch, CURLOPT_URL, $urlItemDetail);
            $responseDetail = curl_exec($ch);
            $itemDetail = json_decode($responseDetail, true);
            $skuv = '';
            
            foreach ($itemDetail['attributes'] ?? [] as $attribute) {
                if ($attribute['id'] === 'SELLER_SKU') {
                    $skuv = $attribute['value_name'] ?? null;
                    break;
                }
            }
            
            if (isset($itemDetail['id'])) {
                $processedItems[] = [
                    'id' => $itemDetail['id'],
                    'title' => $itemDetail['title'],
                    'attributes' => $itemDetail['attributes'],
                    'sku' => $itemDetail['seller_custom_field'] ?? null,
                    'skuv' => $skuv,
                    'price' => $itemDetail['price'],
                    'available_quantity' => $itemDetail['available_quantity'],
                    'permalink' => $itemDetail['permalink'],
                    'status' => $itemDetail['status'],
                    'listing_type_id' => $itemDetail['listing_type_id']
                ];
            }
        }
        
        return $processedItems;
    }
    public function productos()
    {
        if ($this->request->isAJAX()) {
            $start = $this->request->getGet('start') ?? 0;
            $length = $this->request->getGet('length') ?? 10;
            $search = $this->request->getGet('search')['value'] ?? '';
            $order = $this->request->getGet('order')[0] ?? null;

            $columns = ['CodProd', 'Descrip', 'Existen', 'ml_id', 'ultimo_sync'];
            $orderColumn = $columns[$order['column'] ?? 0] ?? 'CodProd';
            $orderDir = $order['dir'] ?? 'ASC';

            $result = $this->mlModel->getProductosConExistenciaPaginado(
                $start,
                $length,
                $search,
                $orderColumn,
                $orderDir
            );

            return $this->response->setJSON([
                'draw' => $this->request->getGet('draw'),
                'recordsTotal' => $result['recordsTotal'],
                'recordsFiltered' => $result['recordsFiltered'],
                'data' => $result['data']
            ]);
        }
    }

    public function sincronizarProducto() {
        if ($this->request->isAJAX()) {
            $mlId = $this->request->getPost('ml_id');
    
            try {
                $accessToken = $this->refreshToken();
                $syncProduct = $this->mlModel->where('ml_id', $mlId)->first();
                
                if (!$syncProduct) {
                    throw new \Exception("Publicación no encontrada");
                }
    
                $producto = $this->productosModel->getProductoConExistencia($syncProduct['cod_prod']);
                if (!$producto) {
                    throw new \Exception("Producto no encontrado en inventario");
                }
    
                $producto['ml_id'] = $mlId;
                $res = $this->actualizarStock($producto, $accessToken, $mlId);
                
                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Producto sincronizado',
                    'data' => $res
                ]);
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }
    }

   
    public function sincronizar() {
        if (!$this->configModel->getConfig('ml_sync_active')['valor']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Sincronización desactivada'
            ]);
        }
    
        try {
            $accessToken = $this->refreshToken();
            $selectedMlIds = $this->request->getPost('productos');
    
            if (empty($selectedMlIds)) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'No hay elementos seleccionados'
                ]);
            }
    
            $results = [];
            foreach ($selectedMlIds as $mlId) {
                $syncProduct = $this->mlModel->where('ml_id', $mlId)->first();
                if ($syncProduct) {
                    $producto = $this->productosModel->getProductoConExistencia($syncProduct['cod_prod']);
                    if ($producto) {
                        $producto['ml_id'] = $mlId;
                        $result = $this->actualizarStock($producto, $accessToken, $mlId);
                        $results[] = $result;
                    }
                }
            }
    
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Sincronización completada',
                'results' => $results
            ]);
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }
    
    public function toggleProductSelection() {
        if ($this->request->isAJAX()) {
            $mlId = $this->request->getPost('ml_id');
            $selected = $this->request->getPost('selected');
            
            try {
                $this->mlModel->toggleSelected($mlId, $selected);
                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Selección actualizada'
                ]);
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }
        return $this->response->setJSON([
            'success' => false,
            'message' => 'Petición no válida'
        ]);
    }
    

    private function refreshToken()
    {
        $clientId = $this->configModel->getConfig('ml_client_id')['valor'];
        $clientSecret = $this->configModel->getConfig('ml_client_secret')['valor'];
        $refreshToken = $this->configModel->getConfig('ml_refresh_token')['valor'];

        $ch = curl_init('https://api.mercadolibre.com/oauth/token');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
            'grant_type' => 'refresh_token',
            'client_id' => $clientId,
            'client_secret' => $clientSecret,
            'refresh_token' => $refreshToken
        ]));

        $response = curl_exec($ch);
        curl_close($ch);

        $data = json_decode($response, true);
        if (isset($data['access_token'])) {
            $this->configModel->setConfig('ml_access_token', $data['access_token']);
            $this->configModel->setConfig('ml_refresh_token', $data['refresh_token']);
            return $data['access_token'];
        }

        throw new \Exception('Error al refrescar el token');
    }

    
    private function actualizarStock($producto, $accessToken, $mlId)
    {
        try {
            // Primero obtenemos el user ID del vendedor
            $urlUser = "https://api.mercadolibre.com/users/me";
            $ch = curl_init($urlUser);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken
            ]);

            $responseUser = curl_exec($ch);
            $userData = json_decode($responseUser, true);

            if (!isset($userData['id'])) {
                throw new \Exception("No se pudo obtener la información del usuario");
            }

            $userId = $userData['id'];
            // Primero intentamos buscar por seller_custom_field (SKU)
            // $urlSearchCustom = "https://api.mercadolibre.com/users/{$userId}/items/search?sku=" . rawurlencode($producto['CodProd']);
            $baseUrl = "https://api.mercadolibre.com/users/{$userId}/items/search";
            $params = [
                'id' => $mlId,
                'sku' => $producto['CodProd']
            ];
            $urlSearchCustom = $baseUrl . '?' . http_build_query($params);
            $ch = curl_init($urlSearchCustom);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken
            ]);
            

            $responseSearch = curl_exec($ch);
            $searchData = json_decode($responseSearch, true);

            // // Si no encontramos por seller_custom_field, intentamos con seller_sku
            if (empty($searchData['results'])) {
                //$urlSearchSku = "https://api.mercadolibre.com/users/{$userId}/items/search?seller_sku=" . rawurlencode($producto['CodProd']);
                $baseUrl = "https://api.mercadolibre.com/users/{$userId}/items/search";
                $params = [
                    'id' => $mlId,
                    'seller_sku' => $producto['CodProd']
                ];
                $urlSearchSku = $baseUrl . '?' . http_build_query($params);
                curl_setopt($ch, CURLOPT_URL, $urlSearchSku);
                $responseSearch = curl_exec($ch);
                $searchData = json_decode($responseSearch, true);
            }

            // // Si no encontramos el producto, registramos el error
            if (empty($searchData['results'])) {
                throw new \Exception("No se encontró el producto con SKU: " . $producto['CodProd']);
            }

            // // Obtenemos el primer resultado (debería ser único por SKU)
            $itemId = $searchData['results'][0];

            // // Obtenemos los detalles del item
            $urlItem = "https://api.mercadolibre.com/items/" . $itemId;
            curl_setopt($ch, CURLOPT_URL, $urlItem);
            $responseItem = curl_exec($ch);
            $itemData = json_decode($responseItem, true);

            if (!isset($itemData['id'])) {
                 throw new \Exception("Error al obtener detalles del producto de ML");
            }

            // // Actualizamos el stock
            $statusPub = $itemData['status'];

            if(intval($producto['Existen']) == 0) {
                $statusPub = 'paused';
            }

            if(intval($producto['Existen']) > 0 && $itemData['status'] == 'paused') {
                $statusPub = 'active';
            }

            // if (!isset($itemInfo['has_bids']) || $itemInfo['has_bids'] === false) {
            //     // Proceder con la actualización
            // } else {
            //     // Manejar el caso de artículos con ofertas de otra manera
            //     // Por ejemplo, crear un nuevo artículo o registrar para actualización posterior
            // }

            if (!isset($itemData['has_bids']) || $itemData['has_bids'] === false) {
                // Proceder con la actualización
                $updateData = [
                    'available_quantity' => (int) $producto['Existen'],
                    'status' => (string) $statusPub
                ];
            } else {
                
                $updateData = [
                    'available_quantity' => (int) $producto['Existen'],
                    'status' => (string) $statusPub
                ];
                // Manejar el caso de artículos con ofertas de otra manera
                // Por ejemplo, crear un nuevo artículo o registrar para actualización posterior
            }
            

            curl_setopt($ch, CURLOPT_URL, $urlItem);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($updateData));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken,
                'Content-Type: application/json'
            ]);

            $responseUpdate = curl_exec($ch);
            $updateData = json_decode($responseUpdate, true);
            curl_close($ch);

            if (isset($updateData['error'])) {
                throw new \Exception("Error al actualizar stock: " . ($updateData['message'] ?? 'Error desconocido'));
            }

            // // Registrar sincronización exitosa
            $mlModel = new \App\Models\MercadoLibreModel();
            $mlModel->registrarSincronizacion(
                $producto['CodProd'],
                $itemData['id'],
                1
            );

            return [
                'success' => true,
                'response' => $updateData,
                'iteminfo' => $itemData,
                'producto' => $producto,
                'message' => 'Stock actualizado correctamente',
                'data' => [
                    'ml_id' => $itemData['id'],
                    'sku' => $producto['CodProd'],
                    'title' => $itemData['title'],
                    'status' => $statusPub,
                    'new_stock' => (int) $producto['Existen']
                ]
            ];

            //return $searchData['results'];

        } catch (\Exception $e) {
            // Registrar error en sincronización
            // if (isset($mlModel)) {
            //     $mlModel->registrarSincronizacion(
            //         $producto['CodProd'],
            //         null,
            //         0
            //     );
            // }

            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
}