<?php

namespace App\Controllers;

use Config\Services;

class Compras extends BaseController
{
    protected $db;
    protected $productoModel;
    protected $proveedorModel;
    
    public function __construct()
    {
        $this->db = \Config\Database::connect();
        // Comprobar si el usuario está logueado
        // if (!session()->get('isLoggedIn')) {
        //     echo view('login');
        //     exit;
        // }
        
        // Cargar los modelos necesarios
        $this->productoModel = new \App\Models\ProductoModel();
        //$this->proveedorModel = new \App\Models\ProveedorModel();
    }

    public function index()
    {
        $data = [
            'proveedores' => [],
        ];
        
        return view('compras/index', $data);
    }
    
    /**
     * Procesar archivo XML DTE
     */
    public function procesarXML()
    {
        if ($this->request->isAJAX()) {
            try {
                $file = $this->request->getFile('xml_file');
                
                if (!$file->isValid()) {
                    throw new \Exception('Archivo XML no válido');
                }
                
                // Leer el contenido del archivo XML
                $xmlContent = file_get_contents($file->getTempName());
                $xml = simplexml_load_string($xmlContent);
                
                if (!$xml) {
                    throw new \Exception('No se pudo procesar el archivo XML');
                }
                
                // Procesar el DTE
                $resultado = $this->procesarDTE($xml);
                
                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Compra procesada correctamente',
                    'compra_id' => $resultado['numeroD']
                ]);
                
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }
    }
    
    /**
     * Procesar el documento DTE y extraer la información
     */
    private function procesarDTE($xml)
    {
        try {
            // Iniciar transacción
            $this->db->transStart();
            
            // Navegar al documento dentro del XML
            $dte = $xml->SetDTE->DTE->Documento;
            $encabezado = $dte->Encabezado;
            
            // Extraer información del emisor (proveedor)
            $emisor = $encabezado->Emisor;
            $rutProveedor = str_replace('-', '', (string)$emisor->RUTEmisor);
            
            // Extraer información del documento
            $idDoc = $encabezado->IdDoc;
            $folio = (string)$idDoc->Folio;
            $fechaEmision = (string)$idDoc->FchEmis;
            $fechaVencimiento = isset($idDoc->FchVenc) ? (string)$idDoc->FchVenc : null;
            
            // Extraer totales
            $totales = $encabezado->Totales;
            $montoNeto = (float)$totales->MntNeto;
            $iva = (float)$totales->IVA;
            $montoTotal = (float)$totales->MntTotal;
            
            // Obtener el correlativo para la compra
            $correlativo = $this->obtenerCorrelativo('PrxComp');
            if (!$correlativo) {
                throw new \Exception('Error al obtener el correlativo para la compra');
            }
            
            // Buscar o crear proveedor
            $codProveedor = $this->obtenerOCrearProveedor($emisor, $rutProveedor);
            
            // Crear el registro en SACOMP
            $compraData = [
                'CodSucu' => '00000',
                'TipoCom' => 'A', // Factura de compra
                'NumeroD' => $correlativo['NumeroD'],
                'CodProv' => $codProveedor,
                'NroUnico' => $correlativo['NroUnico'],
                'Status' => 'A', // Activa
                'NroCtrol' => $folio,
                'CodUsua' => session()->get('user_id') ?? 'ADMIN',
                'Signo' => 1,
                'FechaT' => date('m-d-Y H:i'),
                'FechaE' => $this->convertirFecha($fechaEmision),
                'FechaV' => $fechaVencimiento ? $this->convertirFecha($fechaVencimiento) : null,
                'Moneda' => 'CLP',
                'Factor' => 1,
                'TipoTra' => 'N', // Normal
                'CodUbic' => '01', // Bodega principal
                'Descrip' => (string)$emisor->RznSoc,
                'Direc1' => (string)$emisor->DirOrigen,
                'Ciudad' => (string)$emisor->CiudadOrigen,
                'Monto' => $montoNeto,
                'MtoTax' => $iva,
                'TGravable' => $montoNeto,
                'TExento' => 0,
                'MtoTotal' => $montoTotal,
                'Contado' => $montoTotal,
                'Credito' => 0,
                'SaldoAct' => 0,
                'FechaR' => date('m-d-Y H:i'),
                'Notas1' => 'Importado desde XML DTE'
            ];
            
            // Insertar en SACOMP
            $this->db->table('SACOMP')->insert($compraData);
            
            // Procesar los detalles
            $this->procesarDetallesCompra($dte->Detalle, $compraData);
            
            // Confirmar transacción
            $this->db->transComplete();
            
            if ($this->db->transStatus() === false) {
                $error = $this->db->error();
                $dbError = "Error de BD - Código: " . $error['code'] . " | Mensaje: " . $error['message'];
                throw new \Exception('Error en la transacción de la base de datos: ' . $dbError);
            }
            
            return [
                'numeroD' => $correlativo['NumeroD'],
                'proveedor' => (string)$emisor->RznSoc,
                'total' => $montoTotal
            ];
            
        } catch (\Exception $e) {
            $this->db->transRollback();
            throw $e;
        }
    }
    
    /**
     * Procesar los detalles de la compra
     */
    private function procesarDetallesCompra($detalles, $compraData)
    {
        $nroLinea = 1;
        
        // Si hay un solo detalle, convertirlo a array
        if (!is_array($detalles)) {
            $detalles = [$detalles];
        }
        
        foreach ($detalles as $detalle) {
            // Extraer información del producto
            $codigoItem = '';
            $codigoBarras = '';
            $lote = '';
            
            // Buscar códigos en CdgItem
            if (isset($detalle->CdgItem)) {
                $codigos = is_array($detalle->CdgItem) ? $detalle->CdgItem : [$detalle->CdgItem];
                
                foreach ($codigos as $codigo) {
                    switch ((string)$codigo->TpoCodigo) {
                        case 'INT1':
                            $codigoItem = (string)$codigo->VlrCodigo;
                            break;
                        case 'EAN13':
                            $codigoBarras = (string)$codigo->VlrCodigo;
                            break;
                        case 'LOTE':
                            $lote = (string)$codigo->VlrCodigo;
                            break;
                    }
                }
            }
            
            // Si no hay código interno, usar el NmbItem
            if (empty($codigoItem)) {
                $codigoItem = (string)$detalle->NmbItem;
            }
            
            $cantidad = (float)$detalle->QtyItem;
            $precioUnitario = (float)$detalle->PrcItem;
            $montoItem = (float)$detalle->MontoItem;
            $fechaVencimiento = isset($detalle->FchVencim) ? (string)$detalle->FchVencim : null;
            
            // Buscar o crear el producto
            $this->obtenerOCrearProducto($codigoItem, $detalle, $codigoBarras);
            
            // Crear el registro en SAITEMCOM
            $itemCompraData = [
                'CodSucu' => $compraData['CodSucu'],
                'TipoCom' => $compraData['TipoCom'],
                'NumeroD' => $compraData['NumeroD'],
                'CodProv' => $compraData['CodProv'],
                'Status' => $compraData['Status'],
                'NroLinea' => $nroLinea,
                'CodItem' => $codigoItem,
                'CodUbic' => $compraData['CodUbic'],
                'Descrip1' => (string)$detalle->DscItem,
                'Refere' => $codigoBarras,
                'Signo' => 1,
                'Cantidad' => $cantidad,
                'CantidadO' => $cantidad,
                'ExistAnt' => $this->obtenerExistenciaAnterior($codigoItem, $compraData['CodUbic']),
                'CantidadU' => $cantidad,
                'Costo' => $precioUnitario,
                'TotalItem' => $montoItem,
                'Precio' => $precioUnitario,
                'NroLote' => $lote,
                'FechaE' => date('Y-m-d H:i:s'),
                'FechaV' => $fechaVencimiento ? $this->convertirFecha($fechaVencimiento) : null,
                'EsServ' => 0,
                'EsUnid' => 1,
                'EsPesa' => 0,
                'EsExento' => 0,
                'DEsLote' => !empty($lote) ? 1 : 0,
                'MtoTax' => 0 // Se calcula después si es necesario
            ];
            
            // Insertar en SAITEMCOM
            $this->db->table('SAITEMCOM')->insert($itemCompraData);
            
            // Actualizar inventario
            $this->actualizarInventario($codigoItem, $compraData['CodUbic'], $cantidad, $precioUnitario);
            
            // Crear registro de lote si existe
            if (!empty($lote)) {
                $this->crearRegistroLote($codigoItem, $compraData, $lote, $cantidad, $precioUnitario, $fechaVencimiento);
            }
            
            $nroLinea++;
        }
    }
    
    /**
     * Obtener o crear proveedor
     */
    private function obtenerOCrearProveedor($emisor, $rutProveedor)
    {
        // Buscar proveedor existente por RUT
        $proveedor = $this->db->table('SAPROV')
            ->where('ID3', $rutProveedor)
            ->get()
            ->getRowArray();
        
        if ($proveedor) {
            return $proveedor['CodProv'];
        }
        
        // Crear nuevo proveedor
        $correlativoProv = $this->obtenerCorrelativo('PrxProv');
        if (!$correlativoProv) {
            throw new \Exception('Error al obtener correlativo para proveedor');
        }
        
        $proveedorData = [
            'CodProv' => 'P' . str_pad($correlativoProv['NumeroD'], 6, '0', STR_PAD_LEFT),
            'Descrip' => (string)$emisor->RznSoc,
            'Direc1' => (string)$emisor->DirOrigen,
            'Ciudad' => (string)$emisor->CiudadOrigen,
            'ID3' => $rutProveedor,
            'Activo' => 1,
            'EsProvee' => 1
        ];
        
        $this->db->table('SAPROV')->insert($proveedorData);
        
        return $proveedorData['CodProv'];
    }
    
    /**
     * Obtener o crear producto
     */
    private function obtenerOCrearProducto($codigoItem, $detalle, $codigoBarras = '')
    {
        // Buscar producto existente
        $producto = $this->db->table('SAPROD')
            ->where('CodProd', $codigoItem)
            ->get()
            ->getRowArray();
        
        if ($producto) {
            return $codigoItem;
        }
        
        // Crear nuevo producto
        $productoData = [
            'CodProd' => $codigoItem,
            'Descrip' => (string)$detalle->DscItem,
            'Activo' => 1,
            'Refere' => $codigoBarras,
            'Unidad' => 'UND',
            'CantEmpaq' => 1,
            'Precio1' => (float)$detalle->PrcItem * 1.3, // Precio con margen del 30%
            'CostAct' => (float)$detalle->PrcItem,
            'CostPro' => (float)$detalle->PrcItem,
            'Existen' => 0,
            'EsImport' => 1,
            'ExDecimal' => 0,
            'DEsLote' => 0,
            'DEsVence' => 0
        ];
        
        $this->db->table('SAPROD')->insert($productoData);
        
        return $codigoItem;
    }
    
    /**
     * Crear registro de lote
     */
    private function crearRegistroLote($codigoItem, $compraData, $lote, $cantidad, $costo, $fechaVencimiento)
    {
        $loteData = [
            'CodProd' => $codigoItem,
            'NroUnico' => $compraData['NroUnico'],
            'CodUbic' => $compraData['CodUbic'],
            'NroLote' => $lote,
            'Costo' => $costo,
            'Precio' => $costo * 1.3, // Precio con margen
            'Cantidad' => $cantidad,
            'CantidadU' => $cantidad,
            'FechaE' => date('Y-m-d H:i:s'),
            'FechaV' => $fechaVencimiento ? $this->convertirFecha($fechaVencimiento) : null,
            'CodSucu' => $compraData['CodSucu']
        ];
        
        $this->db->table('SALOTE')->insert($loteData);
    }
    
    /**
     * Actualizar inventario
     */
    private function actualizarInventario($codProd, $codUbic, $cantidad, $costo)
    {
        // Verificar si existe el registro en SAEXIS
        $existencia = $this->db->table('SAEXIS')
            ->where('CodProd', $codProd)
            ->where('CodUbic', $codUbic)
            ->get()
            ->getRowArray();
        
        if ($existencia) {
            // Actualizar la existencia
            $nuevaExistencia = $existencia['Existen'] + $cantidad;
            
            $this->db->table('SAEXIS')
                ->where('CodProd', $codProd)
                ->where('CodUbic', $codUbic)
                ->update(['Existen' => $nuevaExistencia]);
        } else {
            // Crear un nuevo registro
            $this->db->table('SAEXIS')->insert([
                'CodProd' => $codProd,
                'CodUbic' => $codUbic,
                'Existen' => $cantidad,
                'CodSucu' => '00000'
            ]);
        }
        
        // Actualizar también en SAPROD
        $this->db->table('SAPROD')
            ->where('CodProd', $codProd)
            ->set('Existen', 'Existen + ' . $cantidad, false)
            ->set('CostAct', $costo)
            ->update();
    }
    
    /**
     * Obtener existencia anterior
     */
    private function obtenerExistenciaAnterior($codProd, $codUbic)
    {
        $existencia = $this->db->table('SAEXIS')
            ->select('Existen')
            ->where('CodProd', $codProd)
            ->where('CodUbic', $codUbic)
            ->get()
            ->getRowArray();
        
        return $existencia['Existen'] ?? 0;
    }
    
    /**
     * Obtener correlativo
     */
    private function obtenerCorrelativo($tipoDoc)
    {
        $correlativo = $this->db->table('SACORRELSIS')
            ->where('FieldName', $tipoDoc)
            ->get()
            ->getRowArray();
        
        if (!$correlativo) {
            return null;
        }
        
        $nroUnico = intval($correlativo['ValueInt']) + 1;
        
        $this->db->table('SACORRELSIS')
            ->where('FieldName', $tipoDoc)
            ->update(['ValueInt' => $nroUnico]);
        
        return [
            'NroUnico' => $nroUnico,
            'NumeroD' => $nroUnico
        ];
    }
    
    /**
     * Convertir fecha del formato YYYY-MM-DD a MM-DD-YYYY H:i
     */
    private function convertirFecha($fecha)
    {
        $fechaObj = \DateTime::createFromFormat('Y-m-d', $fecha);
        if ($fechaObj) {
            return $fechaObj->format('m-d-Y H:i');
        }
        return date('m-d-Y H:i');
    }
    
    /**
     * Listar compras
     */
    public function listado()
    {
        $compras = $this->db->table('SACOMP')
            ->select('SACOMP.NumeroD, SACOMP.FechaE, SACOMP.CodProv, SAPROV.Descrip as NombreProveedor, SACOMP.MtoTotal, SACOMP.NroCtrol')
            ->join('SAPROV', 'SACOMP.CodProv = SAPROV.CodProv', 'left')
            ->where('SACOMP.TipoCom', 'A')
            ->orderBy('SACOMP.FechaE', 'DESC')
            ->get()
            ->getResultArray();
        
        $data = [
            'compras' => $compras
        ];
        
        return view('compras/listado', $data);
    }
    
    /**
     * Ver detalle de compra
     */
    public function ver($numeroD)
    {
        $compra = $this->db->table('SACOMP')
            ->select('SACOMP.*, SAPROV.Descrip as NombreProveedor')
            ->join('SAPROV', 'SACOMP.CodProv = SAPROV.CodProv', 'left')
            ->where('SACOMP.NumeroD', $numeroD)
            ->where('SACOMP.TipoCom', 'A')
            ->get()
            ->getRowArray();
        
        if (!$compra) {
            return redirect()->to('/compras/listado')->with('error', 'Compra no encontrada');
        }
        
        $detalles = $this->db->table('SAITEMCOM')
            ->where('NumeroD', $numeroD)
            ->where('TipoCom', 'A')
            ->orderBy('NroLinea', 'ASC')
            ->get()
            ->getResultArray();
        
        $data = [
            'compra' => $compra,
            'detalles' => $detalles
        ];
        
        return view('compras/ver', $data);
    }
}