<?php

namespace App\Controllers;

use Dompdf\Dompdf;
use Dompdf\Options;
use Config\Services;

class Facturacion extends BaseController
{
    protected $db;
    protected $productoModel;
    protected $clienteModel;
    protected $emailService;
    
    public function __construct()
    {
        $this->db = \Config\Database::connect();
        // Comprobar si el usuario está logueado
        if (!session()->get('isLoggedIn')) {
            echo view('login');
            exit;
        }
        
        // Cargar los modelos necesarios
        $this->productoModel = new \App\Models\ProductoModel();
        $this->clienteModel = new \App\Models\ClienteModel();
        $this->emailService = Services::emailService();
    }

    public function index()
    {
        $data = [
            'productos' => $this->productoModel->getProductos(),
            'clientes' => $this->clienteModel->getClientes(),
            'vendedores' => $this->obtenerVendedoresR(),
        ];
        
        return view('facturacion/index', $data);
    }
    
    public function crear()
    {
        if ($this->request->isAJAX()) {
            $data = $this->request->getPost();
            
            try {
                // Iniciar transacción
                $this->db->transStart();
                
                // Obtener el correlativo para la cotización
                $correlativo = $this->obtenerCorrelativo('PrxFact');
                if (!$correlativo) {
                    throw new \Exception('Error al obtener el correlativo para la cotización');
                }
                
                // Fecha actual
                $fechaActual = date('Y-m-d H:i:s');
                
                // Datos de la cotización
                $cotizacionData = [
                    'CodSucu' => '00000', // Ajustar según configuración
                    'TipoFac' => 'A', // E para cotizaciones
                    'NumeroD' => $correlativo['NumeroD'],
                    'CodUsua' => session()->get('username'),
                    'Signo' => 1,
                    'FechaT' => date('m-d-Y H:i'),
                    'FechaE' => date('m-d-Y H:i'),
                    'FechaV' => date('m-d-Y H:i', strtotime('+30 days')), // Validez de 30 días
                    'Moneda' => 'CLP', // O la que corresponda
                    'Factor' => 1,
                    'CodClie' => $data['cliente_id'],
                    'Descrip' => $this->clienteModel->getCliente($data['cliente_id'])['Descrip'] ?? 'Cliente',
                    'Monto' => $data['total'],
                    'MtoTotal' => $data['total'],
                    'TGravable' => $data['subtotal'],
                    'TExento' => 0,
                    'MtoTax' => $data['impuesto'],
                    'Notas1' => $data['notas'] ?? ''
                ];
                
                // Insertar en SAFACT
                $this->db->table('SAFACT')->insert($cotizacionData);
                
                // Procesar los items de la cotización
                $items = json_decode($data['items'], true);
                $this->crearDetallesCotizacion($items, $cotizacionData);
                
                // Confirmar transacción
                $this->db->transComplete();
                
                if ($this->db->transStatus() === false) {
                    throw new \Exception('Error en la transacción de la base de datos');
                }

                if (!empty($data['cliente_email'])) {
                //     // Obtener datos completos de la cotización para el PDF
                    $cotizacionCompleta = $this->db->table('SAFACT')
                        ->select('SAFACT.*, SACLIE.Descrip as NombreCliente, SACLIE.ID3 as ClienteID, SACLIE.Direc1, SACLIE.Telef')
                        ->join('SACLIE', 'SAFACT.CodClie = SACLIE.CodClie')
                        ->where('SAFACT.NumeroD', $cotizacionData['NumeroD'])
                        ->where('SAFACT.TipoFac', 'A')
                        ->get()
                        ->getRowArray();
                    
                //     // Obtener detalles de la cotización
                    $detallesCotizacion = $this->db->table('SAITEMFAC')
                        ->where('NumeroD', $cotizacionData['NumeroD'])
                        ->where('TipoFac', 'A')
                        ->orderBy('NroLinea', 'ASC')
                        ->get()
                        ->getResultArray();
                    
                //     // Datos de la empresa
                $empresaQuery = $this->db->table('SACONF')
                ->select('Descrip as nombre, Direc1 as direccion, Telef as telefono, RIF as rif')
                ->get();
            
            $empresaData = $empresaQuery->getRowArray();
            
            // Verificar si se obtuvieron los datos, si no, usar valores por defecto
            if (!$empresaData) {
                $empresaData = [
                    'nombre' => 'NOMBRE DE TU EMPRESA S.A.',
                    'direccion' => 'Dirección de la empresa',
                    'telefono' => 'Teléfono de la empresa',
                    'nit' => 'RIF de la empresa'
                ];
            } else {
                // Limpiar datos si es necesario (eliminar espacios extra, etc.)
                $empresaData = [
                    'nombre' => trim($empresaData['nombre'] ?? 'NOMBRE DE TU EMPRESA S.A.'),
                    'direccion' => trim($empresaData['direccion'] ?? 'Dirección de la empresa'),
                    'telefono' => trim($empresaData['telefono'] ?? 'Teléfono de la empresa'),
                    'nit' => trim($empresaData['rif'] ?? 'RIF de la empresa')
                ];
            }
                    
                //     // Enviar email con el PDF
                    $emailService = Services::emailService();
                    $emailSent = $emailService->sendCotizacionEmail(
                        $data['cliente_email'],
                        $cotizacionCompleta,
                        $detallesCotizacion,
                        $empresaData
                    );
                    
                    if (!$emailSent) {
                        log_message('error', 'No se pudo enviar el correo para la cotización #' . $cotizacionData['NumeroD']);
                    }
                }
                
                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Factura creada correctamente',
                    'cotizacion_id' => $cotizacionData['NumeroD']
                ]);
                
            } catch (\Exception $e) {
                $this->db->transRollback();
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }
    }
    
    /**
     * Obtiene el siguiente correlativo para la cotización
     */
    private function obtenerCorrelativo($tipoDoc)
    {
        // Buscar el correlativo actual
        $correlativo = $this->db->table('SACORRELSIS')
            ->where('FieldName', $tipoDoc)
            ->get()
            ->getRowArray();
        
        if (!$correlativo) {
            return null;
        }
        
        // Obtener el siguiente número
        $nroUnico = intval($correlativo['ValueInt']) + 1;
        
        // Actualizar el correlativo
        $this->db->table('SACORRELSIS')
            ->where('FieldName', $tipoDoc)
            ->update(['ValueInt' => $nroUnico]);
        
        return [
            'NroUnico' => $nroUnico,
            'NumeroD' => $nroUnico
        ];
    }
    
    /**
     * Crea los detalles de la cotización
     */
    private function crearDetallesCotizacion($items, $cotizacionData)
    {
        $nroLinea = 1;
        
        foreach ($items as $item) {
            $sku = $item['sku'];
            
            // Obtener datos del producto
            $producto = $this->db->table('SAPROD')
                ->where('CodProd', $sku)
                ->get()
                ->getRowArray();
            
            if (!$producto) {
                continue; // Saltar este item si no se encuentra el producto
            }
            
            // Calcular precios
            $cantidad = $item['cantidad'];
            $precioUnitario = $item['precio'];
            $subtotal = $cantidad * $precioUnitario;
            $impuesto = $subtotal * 0.19; // 19% IVA
            
            // Preparar los datos del ítem de cotización
            $itemCotizacionData = [
                'CodSucu' => $cotizacionData['CodSucu'],
                'TipoFac' => $cotizacionData['TipoFac'],
                'NumeroD' => $cotizacionData['NumeroD'],
                'NroLinea' => $nroLinea,
                'CodItem' => $sku,
                'CodUbic' => '01', // Ajustar según corresponda
                'Descrip1' => $producto['Descrip'] ?? $item['nombre'],
                'Descrip2' => $producto['Descrip2'] ?? '',
                'Descrip3' => $producto['Descrip3'] ?? '',
                'Refere' => $producto['Refere'] ?? '',
                'Signo' => 1,
                'Cantidad' => $cantidad,
                'ExistAnt' => 0, // No afecta inventario
                'Costo' => $producto['CostPro'] ?? 0,
                'Precio' => $precioUnitario,
                'MtoTax' => $impuesto,
                'TotalItem' => $subtotal + $impuesto,
                'FechaE' => date('Y-m-d H:i:s'),
                'EsServ' => 0,
                'EsExento' => 0
            ];

            // Obtener existencia actual
            $existencia = $this->obtenerExistenciaAnterior($itemCotizacionData['CodItem'], $itemCotizacionData['CodUbic']);
                        
            // Insertar el ítem de cotización
            $this->db->table('SAITEMFAC')->insert($itemCotizacionData);
            // Actualizar inventario
            $this->actualizarInventario($itemCotizacionData['CodItem'], $itemCotizacionData['CodUbic'], $itemCotizacionData['Cantidad']);

            $nroLinea++;
        }
    }
    
    // public function buscarProductos()
    // {
    //     if ($this->request->isAJAX()) {
    //         $term = $this->request->getPost('term');
            
    //         $productos = $this->db->table('SAPROD')
    //             ->select('CodProd, Descrip, Descrip2, Precio1')
    //             ->like('Descrip', $term)
    //             ->orLike('Descrip2', $term)
    //             ->orLike('CodProd', $term)
    //             ->limit(10)
    //             ->get()
    //             ->getResultArray();
            
    //         return $this->response->setJSON($productos);
    //     }
    // }
    public function obtenerVendedores()
    {
        try {
            $vendedores = $this->db->table('SAVEND')
                ->select('CodVend, Descrip')
                ->where('Activo', 1)
                ->orderBy('Descrip', 'ASC')
                ->get()
                ->getResultArray();
            
            return $this->response->setJSON($vendedores);
        } catch (\Exception $e) {
            return $this->response->setJSON([]);
        }
    }
    public function obtenerVendedoresR()
    {
        try {
            $vendedores = $this->db->table('SAVEND')
                ->select('CodVend, Descrip')
                ->where('Activo', 1)
                ->orderBy('Descrip', 'ASC')
                ->get()
                ->getResultArray();
            
            return $vendedores;
        } catch (\Exception $e) {
            return [];
        }
    }
    public function buscarProductos()
    {
        if ($this->request->isAJAX()) {
            $term = $this->request->getPost('term');
            
            $query = $this->db->table('SAPROD p')
                ->select('p.CodProd, p.Descrip, p.Descrip2, p.Refere, p.Precio1, p.Precio2, p.Precio3, SUM(e.Existen) as Existen')
                ->join('SAEXIS e', 'p.CodProd = e.CodProd', 'left')
                ->groupBy('p.CodProd, p.Descrip, p.Descrip2, p.Refere, p.Precio1, p.Precio2, p.Precio3');
                
            if (!empty($term)) {
                $query->groupStart()
                    ->like('CodProd', $term)
                    ->orLike('Descrip', $term)
                    ->orLike('Descrip2', $term)
                    ->orLike('Refere', $term)
                    ->groupEnd();
            }
            
            $productos = $query->limit(50)
                ->get()
                ->getResultArray();
            
            return $this->response->setJSON($productos);
        }
        
        return $this->response->setJSON([]);
    }
    public function listado()
    {
        $facturas = $this->db->table('SAFACT')
            ->select('SAFACT.NumeroD, SAFACT.FechaE, SAFACT.CodClie, SACLIE.Descrip as NombreCliente, SAFACT.MtoTotal')
            ->join('SACLIE', 'SAFACT.CodClie = SACLIE.CodClie')
            ->where('SAFACT.TipoFac', 'A') // Solo cotizaciones
            ->orderBy('SAFACT.FechaE', 'DESC')
            ->get()
            ->getResultArray();
        
        $data = [
            'facturas' => $facturas
        ];
        
        return view('facturacion/listado', $data);
    }
    public function listado_facturas()
    {
        $facturas = $this->db->table('SAFACT')
            ->select('SAFACT.NumeroD, SAFACT.FechaE, SAFACT.CodClie, SACLIE.Descrip as NombreCliente, SAFACT.MtoTotal')
            ->join('SACLIE', 'SAFACT.CodClie = SACLIE.CodClie')
            ->where('SAFACT.TipoFac', 'A') // Solo cotizaciones
            ->orderBy('SAFACT.FechaE', 'DESC')
            ->get()
            ->getResultArray();
        
        $data = [
            'facturas' => $facturas
        ];
        
        return view('facturacion/listado_facturas', $data);
    }
    public function pdf($numeroD)
    {
        // Obtener encabezado de la cotización
        $cotizacion = $this->db->table('SAFACT')
            ->select('SAFACT.*, SACLIE.Descrip as NombreCliente, SACLIE.ID3 as ClienteID, SACLIE.Direc1, SACLIE.Telef')
            ->join('SACLIE', 'SAFACT.CodClie = SACLIE.CodClie')
            ->where('SAFACT.NumeroD', $numeroD)
            ->where('SAFACT.TipoFac', 'A')
            ->get()
            ->getRowArray();
        
        if (!$cotizacion) {
            return redirect()->to('/facturacion/listado')->with('error', 'Cotización no encontrada');
        }
        
        // Obtener detalle de la cotización
        $detalles = $this->db->table('SAITEMFAC')
            ->where('NumeroD', $numeroD)
            ->where('TipoFac', 'A')
            ->orderBy('NroLinea', 'ASC')
            ->get()
            ->getResultArray();
        
        // Obtener datos de la empresa
        $empresaQuery = $this->db->table('SACONF')
    ->select('Descrip as nombre, Direc1 as direccion, Telef as telefono, RIF as rif')
    ->get();

$empresaData = $empresaQuery->getRowArray();

// Verificar si se obtuvieron los datos, si no, usar valores por defecto
if (!$empresaData) {
    $empresaData = [
        'nombre' => 'NOMBRE DE TU EMPRESA S.A.',
        'direccion' => 'Dirección de la empresa',
        'telefono' => 'Teléfono de la empresa',
        'nit' => 'RIF de la empresa'
    ];
} else {
    // Limpiar datos si es necesario (eliminar espacios extra, etc.)
    $empresaData = [
        'nombre' => trim($empresaData['nombre'] ?? 'NOMBRE DE TU EMPRESA S.A.'),
        'direccion' => trim($empresaData['direccion'] ?? 'Dirección de la empresa'),
        'telefono' => trim($empresaData['telefono'] ?? 'Teléfono de la empresa'),
        'nit' => trim($empresaData['rif'] ?? 'RIF de la empresa')
    ];
}
        
        $data = [
            'factura' => $cotizacion,
            'detalles' => $detalles,
            'empresa' => $empresa
        ];
        
        $dompdf = new Dompdf();
        
        // Renderizar vista HTML para PDF
        $html = view('facturacion/pdf_template', $data);
        
        // Cargar HTML en DOMPDF
        $dompdf->loadHtml($html);
        
        // Configurar tamaño de papel y orientación
        $dompdf->setPaper('A4', 'portrait');
        
        // Renderizar PDF
        $dompdf->render();
        
        // Salida del PDF
        return $dompdf->stream('Factura_' . $numeroD . '.pdf', ['Attachment' => false]);
    }
    public function ver($numeroD)
    {
        // Obtener encabezado de la cotización
        $cotizacion = $this->db->table('SAFACT')
            ->select('SAFACT.*, SACLIE.Descrip as NombreCliente, SACLIE.ID3 as ClienteID')
            ->join('SACLIE', 'SAFACT.CodClie = SACLIE.CodClie')
            ->where('SAFACT.NumeroD', $numeroD)
            ->where('SAFACT.TipoFac', 'A')
            ->get()
            ->getRowArray();
        
        if (!$cotizacion) {
            return redirect()->to('/facturacion/listado')->with('error', 'Cotización no encontrada');
        }
        
        // Obtener detalle de la cotización
        $detalles = $this->db->table('SAITEMFAC')
            ->where('NumeroD', $numeroD)
            ->where('TipoFac', 'A')
            ->orderBy('NroLinea', 'ASC')
            ->get()
            ->getResultArray();
        
        $data = [
            'factura' => $cotizacion,
            'detalles' => $detalles
        ];
        
        return view('facturacion/ver', $data);
    }
    // public function buscarClientes()
    // {
    //     if ($this->request->isAJAX()) {
    //         $term = $this->request->getPost('term');
            
    //         $clientes = $this->db->table('SACLIE')
    //             ->select('CodClie, Descrip, ID3')
    //             ->like('Descrip', $term)
    //             ->orLike('ID3', $term)
    //             ->orLike('CodClie', $term)
    //             ->limit(10)
    //             ->get()
    //             ->getResultArray();
            
    //         return $this->response->setJSON($clientes);
    //     }
    // }
    public function buscarClientes()
    {
        if ($this->request->isAJAX()) {
            $term = $this->request->getPost('term');
            
            $query = $this->db->table('SACLIE')
                ->select('CodClie, Descrip, ID3, Email');
                
            if (!empty($term)) {
                $query->groupStart()
                    ->like('CodClie', $term)
                    ->orLike('Descrip', $term)
                    ->orLike('ID3', $term)
                    ->groupEnd();
            }
            
            $clientes = $query->limit(10000)
                ->get()
                ->getResultArray();
            
            return $this->response->setJSON($clientes);
        }
        
        return $this->response->setJSON([]);
    }

    public function convertir($numeroD)
{
    // Verificar si la cotización existe
    $cotizacion = $this->db->table('SAFACT')
        ->where('NumeroD', $numeroD)
        ->where('TipoFac', 'E')
        ->get()
        ->getRowArray();
    
    if (!$cotizacion) {
        return redirect()->to('/cotizaciones/listado')->with('error', 'Cotización no encontrada');
    }
    
    // Obtener detalles de la cotización
    $detalles = $this->db->table('SAITEMFAC')
        ->where('NumeroD', $numeroD)
        ->where('TipoFac', 'E')
        ->get()
        ->getResultArray();
    
    try {
        // Iniciar transacción
        $this->db->transStart();
        
        // Obtener el correlativo para la factura
        $correlativo = $this->obtenerCorrelativo('PrxFact');
        if (!$correlativo) {
            throw new \Exception('Error al obtener el correlativo para la factura');
        }
        
        // Crear la factura
        $facturaData = $cotizacion;
        $facturaData['TipoFac'] = 'A'; // A para facturas
        $facturaData['NumeroD'] = $correlativo['NumeroD'];
        $facturaData['FechaE'] = date('m-d-Y H:i');
        $facturaData['FechaR'] = date('m-d-Y H:i');
        $facturaData['FechaV'] = date('m-d-Y H:i', strtotime('+30 days'));
        $facturaData['Notas1'] = 'Generada desde cotización #' . $numeroD . '. ' . ($facturaData['Notas1'] ?? '');
        
        // Eliminar campo que no queremos duplicar
        unset($facturaData['id']);
        
        // Insertar en SAFACT
        $this->db->table('SAFACT')->insert($facturaData);
        
        // Procesar detalles de la factura
        foreach ($detalles as $detalle) {
            // Obtener existencia actual
            $existencia = $this->obtenerExistenciaAnterior($detalle['CodItem'], $detalle['CodUbic']);
            
            // Modificar datos para la factura
            $detalle['TipoFac'] = 'A';
            $detalle['NumeroD'] = $correlativo['NumeroD'];
            $detalle['ExistAnt'] = $existencia;
            $detalle['FechaE'] = date('Y-m-d H:i:s');
            
            // Eliminar campo que no queremos duplicar
            unset($detalle['id']);
            
            // Insertar en SAITEMFAC
            $this->db->table('SAITEMFAC')->insert($detalle);
            
            // Actualizar inventario
            $this->actualizarInventario($detalle['CodItem'], $detalle['CodUbic'], $detalle['Cantidad']);
        }
        
        // Confirmar transacción
        $this->db->transComplete();
        
        if ($this->db->transStatus() === false) {
            throw new \Exception('Error en la transacción de la base de datos');
        }
        
        return redirect()->to('/cotizaciones/listado')->with('message', 'Cotización convertida a factura exitosamente. Número de factura: ' . $correlativo['NumeroD']);
        
    } catch (\Exception $e) {
        return redirect()->to('/cotizaciones/listado')->with('error', 'Error al convertir la cotización: ' . $e->getMessage());
    }
}

/**
 * Obtener existencia anterior
 */
private function obtenerExistenciaAnterior($codProd, $codUbic)
{
    $existencia = $this->db->table('SAEXIS')
        ->select('Existen')
        ->where('CodProd', $codProd)
        ->where('CodUbic', $codUbic)
        ->get()
        ->getRowArray();
    
    return $existencia['Existen'] ?? 0;
}

/**
 * Actualizar inventario
 */
private function actualizarInventario($codProd, $codUbic, $cantidad)
{
    // Verificar si existe el registro en SAEXIS
    $existencia = $this->db->table('SAEXIS')
        ->where('CodProd', $codProd)
        ->where('CodUbic', $codUbic)
        ->get()
        ->getRowArray();
    
    if ($existencia) {
        // Actualizar la existencia
        $nuevaExistencia = $existencia['Existen'] - $cantidad;
        
        $this->db->table('SAEXIS')
            ->where('CodProd', $codProd)
            ->where('CodUbic', $codUbic)
            ->update(['Existen' => $nuevaExistencia]);
    } else {
        // Crear un nuevo registro
        $this->db->table('SAEXIS')->insert([
            'CodProd' => $codProd,
            'CodUbic' => $codUbic,
            'Existen' => -$cantidad // Existencia negativa, indica que se vendió sin stock
        ]);
    }
}
}