<?php

namespace App\Controllers;

class Clientes extends BaseController
{
    protected $clienteModel;
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
        $this->clienteModel = new \App\Models\ClienteModel();
        
        // Comprobar si el usuario está logueado
        if (!session()->get('isLoggedIn')) {
            echo view('login');
            exit;
        }
    }

    public function index()
    {
        $data = [
            'clientes' => $this->clienteModel->getClientes(50)
        ];
        
        return view('clientes/index', $data);
    }
    
    public function crear()
    {
        // Obtener el último código para mostrar como referencia
        $ultimoCodigo = $this->db->table('SACLIE')
            ->selectMax('CodClie')
            ->where('CodClie LIKE', 'C%')
            ->get()
            ->getRowArray();
            
        $ultimoNumero = 0;
        if ($ultimoCodigo && isset($ultimoCodigo['CodClie'])) {
            $ultimoNumero = intval(substr($ultimoCodigo['CodClie'], 1));
        }
        $proximoCodigo = 'C' . str_pad(($ultimoNumero + 1), 5, '0', STR_PAD_LEFT);
        
        $data = [
            'ultimo_codigo' => $proximoCodigo,
            'paises' => $this->getPaises(),
            'estados' => $this->getEstadosPorDefecto(), // Cargar estados por defecto (ej: Colombia)
            'ciudades' => [],
            'municipios' => [],
        ];
        
        return view('clientes/crear', $data);
    }
    
    public function guardar()
    {
        $data = $this->request->getPost();
        
        // Validación básica
        $rules = [
            'descrip' => 'required',
            'id3' => 'required',
            'pais' => 'required|integer',
            'estado' => 'required|integer',
            'ciudad' => 'required|integer'
        ];
        
        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }
        
        // Preparar datos para insertar
        $clienteData = [
            'CodClie' => $data['codclie'],
            'Descrip' => $data['descrip'],
            'ID3' => $data['id3'],
            'TipoID3' => $data['tipoid3'] ?? 1,
            'Activo' => isset($data['activo']) ? 1 : 0,
            'Represent' => $data['represent'] ?? '',
            'Direc1' => $data['direc1'] ?? '',
            'Direc2' => $data['direc2'] ?? '',
            //'Pais' => $data['pais'] ?? '',
            //'Estado' => $data['estado'] ?? '',
            //'Ciudad' => $data['ciudad'] ?? '',
            //'Municipio' => $data['municipio'] ?? '',
            'Pais' => $data['pais'],
            'Estado' => $data['estado'], 
            'Ciudad' => $data['ciudad'],
            'ZipCode' => $data['zipcode'] ?? '',
            'Telef' => $data['telef'] ?? '',
            'Movil' => $data['movil'] ?? '',
            'Email' => $data['email'] ?? '',
            'Fax' => $data['fax'] ?? '',
            'FechaE' => date('Y-m-d'),
            'TipoCli' => $data['tipocli'] ?? 1,
            'TipoPVP' => $data['tipopvp'] ?? 1,
            'EsCredito' => isset($data['escredito']) ? 1 : 0,
            'Clase' => $data['clase'] ?? ''
        ];
        
        try {
            // Insertar en la base de datos
            $result = $this->db->table('SACLIE')->insert($clienteData);
            
            if (!$result) {
                return redirect()->back()->withInput()->with('error', 'Error al insertar el cliente');
            }
            
            return redirect()->to('clientes')->with('message', 'Cliente creado correctamente');
            
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', $e->getMessage());
        }
    }
    /**
     * Obtener países desde la tabla SAPAIS
     */
    private function getPaises()
    {
        try {
            $query = $this->db->query("SELECT TOP (1000) [Descrip], [Pais] FROM [Ftodo].[dbo].[SAPAIS] ORDER BY [Descrip]");
            return $query->getResultArray();
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener países: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Obtener estados por país via AJAX
     */
    public function getEstados()
    {
        $paisId = $this->request->getVar('pais_id');
        
        if (!$paisId) {
            return $this->response->setJSON(['error' => 'País no especificado']);
        }
        
        try {
            $query = $this->db->query(
                "SELECT TOP (1000) [Descrip], [Estado], [Pais] FROM [Ftodo].[dbo].[SAESTADO] WHERE [Pais] = ? ORDER BY [Descrip]", 
                [$paisId]
            );
            $estados = $query->getResultArray();
            
            return $this->response->setJSON($estados);
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener estados: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Error al cargar estados']);
        }
    }
    
    /**
     * Obtener ciudades por estado via AJAX
     */
    public function getCiudades()
    {
        $estadoId = $this->request->getVar('estado_id');
        
        if (!$estadoId) {
            return $this->response->setJSON(['error' => 'Estado no especificado']);
        }
        
        try {
            $query = $this->db->query(
                "SELECT TOP (1000) [Descrip], [Ciudad], [Estado] FROM [Ftodo].[dbo].[SACIUDAD] WHERE [Estado] = ? ORDER BY [Descrip]", 
                [$estadoId]
            );
            $ciudades = $query->getResultArray();
            
            return $this->response->setJSON($ciudades);
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener ciudades: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Error al cargar ciudades']);
        }
    }
    
    /**
     * Obtener municipios por ciudad via AJAX (opcional)
     */
    public function getMunicipios()
    {
        $ciudadId = $this->request->getVar('ciudad_id');
        
        if (!$ciudadId) {
            return $this->response->setJSON(['error' => 'Ciudad no especificada']);
        }
        
        try {
            // Si tienes una tabla de municipios, usa esta consulta
            // Sino, puedes retornar la misma ciudad como municipio
            $municipios = [
                ['Descrip' => 'Centro', 'Municipio' => $ciudadId . '_CENTRO'],
                ['Descrip' => 'Norte', 'Municipio' => $ciudadId . '_NORTE'],
                ['Descrip' => 'Sur', 'Municipio' => $ciudadId . '_SUR'],
                ['Descrip' => 'Oriente', 'Municipio' => $ciudadId . '_ORIENTE'],
                ['Descrip' => 'Occidente', 'Municipio' => $ciudadId . '_OCCIDENTE']
            ];
            
            return $this->response->setJSON($municipios);
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener municipios: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Error al cargar municipios']);
        }
    }
    
    /**
     * Obtener estados por defecto (Colombia)
     */
    private function getEstadosPorDefecto($paisId = 2) // Asumiendo que Colombia tiene ID 2
    {
        try {
            $query = $this->db->query(
                "SELECT TOP (1000) [Descrip], [Estado], [Pais] FROM [Ftodo].[dbo].[SAESTADO] WHERE [Pais] = ? ORDER BY [Descrip]", 
                [$paisId]
            );
            return $query->getResultArray();
        } catch (\Exception $e) {
            log_message('error', 'Error al obtener estados por defecto: ' . $e->getMessage());
            return [];
        }
    }
}